; Cooling Fan controller for PWM fans and
; Loudspeaker protector for amplifiers

	ERRORLEVEL -302
	ERRORLEVEL -306

	list P=16F1459
	#include p16f1459.inc

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _FOSC_INTOSC & _WDTE_OFF & _PWRTE_ON & _MCLRE_OFF & _CP_OFF & _BOREN_OFF & _CLKOUTEN_OFF &_IESO_OFF & _FCMEN_OFF

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _WRT_OFF & _CPUDIV_NOCLKDIV & _USBLSCLK_24MHz & _PLLMULT_3x & _PLLEN_DISABLED & _STVREN_OFF & _BORV_HI & _LPBOR_OFF & _LVP_OFF 


; Define variables at memory locations

; Bank 0 RAM

NTC_TH1		equ	H'20'  	; thermistor 1 degrees C from table (0 to 70 degrees C)
NTC_TH2		equ	H'21'	; thermistor 2 degrees C from table (0 to 70 degrees C)
THRESH		equ	H'22'	; threshold value converted to degrees C
RANGE		equ	H'23'	; temperature range deg C
OVER_T		equ	H'24'	; over temperature Minimum is THRESH plus RANGE	
AC_DET		equ	H'25'	; AC detect value
AMP1_OFF	equ	H'26'	; amp1 offset
AMP2_OFF	equ	H'27'	; amp2 offset	
OFF_AC		equ	H'28'	; AC detection relay off flag	
OFFS_1P		equ	H'29'	; amp1 offset flag positive detect
OFFS_1N		equ	H'2A'	; amp1 offset flag negative detect
OFF_OFFS1	equ	H'2B'	; main offset flag for offset/amplifier 1/ relay off
OFFS_2P		equ	H'2C'	; amp2 offset flag positive detect
OFFS_2N		equ	H'2D'	; amp2 offset flag negative detect
OFF_OFFS2	equ	H'2E'	; main offset flag for offset/amplifier 2/ relay off
TIMEOUT		equ	H'2F'	; initial power up timer. Flag set after so relay can be on subject to AC and offset flags above
T_ALM		equ	H'30'	; temperature alarm on flag
DUTY_X		equ	H'31'	; fan drive duty cycle based on temperature
COUNT1		equ	H'32'	; count down timer ls byte
COUNT2		equ	H'33'	; count down timer ms byte
FAN_RUN		equ	H'34'	; fan run counter

; math routines

TEMPB0		equ	H'5D'
TEMPB1		equ	H'5E'
TEMPB2		equ	H'5F'
TEMP		equ H'60'
REMB3		equ H'61'
REMB2		equ	H'62'
REMB1      	equ H'63'
REMB0		equ	H'64'
AARGB5		equ	H'65'
AARGB4      equ H'66'
AARGB3		equ	H'67'
AARGB2      equ H'68'
AARGB1      equ H'69'
AARGB0      equ H'6A'	; most significant byte of argument A
BARGB3      equ H'6B'
BARGB2      equ H'6C'
BARGB1      equ H'6D'
BARGB0      equ H'6E'	; most significant byte of argument B
LOOPCOUNT   equ H'6F'  	; division counter

; All banks

A_D_LOW		equ	H'70'	; least significant bits for A/D conversion
A_D_HI		equ	H'71'	; ms byte
READADDMS	equ	H'72'	; flash memory read address ms byte making up a 14 bit byte (ms byte has 6 bits)
READADDLS	equ	H'73'  	; flash memory read address ls byte (ls byte has 8 bits)
DATA_ADD	equ	H'74'	; data address
TEMPS		equ	H'75'	; temporary 
OFFS_COUNT1 equ	H'76'	; offset detect counter for channel 1
OFFS_COUNT2 equ	H'77'	; offset detect counter for channel 2

; Reserved space for DATA. Data used in the ls bytes only. Written to program memory starting at H0700 
FANS0		equ	H'78'	; FANS connected ms byte
FANS		equ	H'79'	; FANS connected ls byte
MIN_PWM0	equ	H'7A'	; ms byte minimum PWM to run fans
MIN_PWM		equ	H'7B'	; ls byte minimum PWM to run fans 

; initial values

	org	H'700'		; start address of where data memory is stored for power setting
	DA				H'3F00' ; fans connected initially 0
	DA				H'3F00' ; min pwm, initially 0


; define reset and interrupt vector start addresses

	org		0  				; start at address 0000h
	goto	MAIN
	org     4				; interrupt vector 0004h, start interrupt routine here
	goto	INTERRUPT

;***************************************************************************************


; Lookup table for temperature
TEMP_CALC

; return value is directly equivalent to compare value on trim pot. 
; degrees C based on A/D value from voltage divider with 10k and thermistor.
; (NTC thermistor is 10k at 25 deg C and with a Beta of 3960)

; check for >195 for end of table length so return 0 if >
	movwf	TEMP
	sublw	D'195'
	btfss	STATUS,C	; if C is clear then >195 so return 0
	retlw	D'0'		; zero degrees
	movf	TEMP,w
 	brw
	retlw	D'100'		; for A/D of 0
	retlw	D'100'		; for A/D of 1
	retlw	D'100'		; for A/D of 2
	retlw	D'100'		; for A/D of 3
	retlw	D'100'		; for A/D of 4
	retlw	D'100'		; for A/D of 5
	retlw	D'100'		; for A/D of 6
	retlw	D'100'		; for A/D of 7
	retlw	D'100'		; for A/D of 8
	retlw	D'100'		; for A/D of 9
	retlw	D'100'		; for A/D of 10
	retlw	D'100'		; for A/D of 11
	retlw	D'100'		; for A/D of 12
	retlw	D'100'		; for A/D of 13
	retlw	D'100'		; for A/D of 14
	retlw	D'100'		; for A/D of 15
	retlw	D'100'		; for A/D of 16
	retlw	D'99'		; for A/D of 17
	retlw	D'97'		; for A/D of 18
	retlw	D'95'		; for A/D of 19
	retlw	D'93'		; for A/D of 20
	retlw	D'91'		; for A/D of 21
	retlw	D'90'		; for A/D of 22
	retlw	D'88'		; for A/D of 23
	retlw	D'86'		; for A/D of 24
	retlw	D'85'		; for A/D of 25
	retlw	D'84'		; for A/D of 26
	retlw	D'83'		; for A/D of 27
	retlw	D'81'		; for A/D of 28
	retlw	D'80'		; for A/D of 29
	retlw	D'78'		; for A/D of 30
	retlw	D'77'		; for A/D of 31
	retlw	D'76'		; for A/D of 32
	retlw	D'75'		; for A/D of 33
	retlw	D'74'		; for A/D of 34
	retlw	D'73'		; for A/D of 35
	retlw	D'72'		; for A/D of 36
	retlw	D'71'		; for A/D of 37
	retlw	D'70'		; for A/D of 38, 70 degrees C. 
	retlw	D'69'		; for A/D of 39, 69 degrees C. 
	retlw	D'68'		; for A/D of 40, 68 degrees C. 
	retlw	D'67'		; for A/D of 41, 67 degrees C. 
	retlw	D'67'		; for A/D of 42, 67 degrees C. 
	retlw	D'66'		; for A/D of 43, 66 degrees C. 
	retlw	D'65'		; for A/D of 44, 65 degrees C. 
	retlw	D'64'		; for A/D of 45, 64 degrees C. 
	retlw	D'63'		; for A/D of 46, 63 degrees C. 
	retlw	D'62'		; for A/D of 47, 62 degrees C. 
	retlw	D'62'		; for A/D of 48, 62 degrees C. 
	retlw	D'61'		; for A/D of 49, 61 degrees C. 
	retlw	D'60'		; for A/D of 50
	retlw	D'59'		; for A/D of 51
	retlw	D'59'		; for A/D of 52
	retlw	D'58'		; for A/D of 53
	retlw	D'57'		; for A/D of 54
	retlw	D'57'		; for A/D of 55
	retlw	D'56'		; for A/D of 56
	retlw	D'55'		; for A/D of 57
	retlw	D'55'		; for A/D of 58
	retlw	D'54'		; for A/D of 59
	retlw	D'54'		; for A/D of 60
	retlw	D'53'		; for A/D of 61
	retlw	D'52'		; for A/D of 62
	retlw	D'52'		; for A/D of 63
	retlw	D'51'		; for A/D of 64
	retlw	D'51'		; for A/D of 65
	retlw	D'50'		; for A/D of 66
	retlw	D'50'		; for A/D of 67
	retlw	D'49'		; for A/D of 68
	retlw	D'49'		; for A/D of 69
	retlw	D'48'		; for A/D of 70
	retlw	D'48'		; for A/D of 71
	retlw	D'47'		; for A/D of 72
	retlw	D'47'		; for A/D of 73
	retlw	D'46'		; for A/D of 74
	retlw	D'46'		; for A/D of 75
	retlw	D'45'		; for A/D of 76
	retlw	D'45'		; for A/D of 77
	retlw	D'44'		; for A/D of 78
	retlw	D'44'		; for A/D of 79
	retlw	D'43'		; for A/D of 80
	retlw	D'43'		; for A/D of 81
	retlw	D'42'		; for A/D of 82
	retlw	D'42'		; for A/D of 83
	retlw	D'41'		; for A/D of 84
	retlw	D'41'		; for A/D of 85
	retlw	D'41'		; for A/D of 86
	retlw	D'40'		; for A/D of 87
	retlw	D'40'		; for A/D of 88
	retlw	D'39'		; for A/D of 89
	retlw	D'39'		; for A/D of 90
	retlw	D'38'		; for A/D of 91
	retlw	D'38'		; for A/D of 92
	retlw	D'38'		; for A/D of 93
	retlw	D'37'		; for A/D of 94
	retlw	D'37'		; for A/D of 95
	retlw	D'36'		; for A/D of 96
	retlw	D'36'		; for A/D of 97
	retlw	D'36'		; for A/D of 98
	retlw	D'35'		; for A/D of 99
	retlw	D'35'		; for A/D of 100
	retlw	D'34'		; for A/D of 101
	retlw	D'34'		; for A/D of 102
	retlw	D'34'		; for A/D of 103
	retlw	D'33'		; for A/D of 104
	retlw	D'33'		; for A/D of 105
	retlw	D'32'		; for A/D of 106
	retlw	D'32'		; for A/D of 107
	retlw	D'32'		; for A/D of 108
	retlw	D'31'		; for A/D of 109
	retlw	D'31'		; for A/D of 110
	retlw	D'31'		; for A/D of 111
	retlw	D'30'		; for A/D of 112
	retlw	D'30'		; for A/D of 113
	retlw	D'29'		; for A/D of 114
	retlw	D'29'		; for A/D of 115
	retlw	D'29'		; for A/D of 116
	retlw	D'28'		; for A/D of 117
	retlw	D'28'		; for A/D of 118
	retlw	D'28'		; for A/D of 119
	retlw	D'27'		; for A/D of 120
	retlw	D'27'		; for A/D of 121
	retlw	D'27'		; for A/D of 122
	retlw	D'26'		; for A/D of 123
	retlw	D'26'		; for A/D of 124
	retlw	D'26'		; for A/D of 125
	retlw	D'25'		; for A/D of 126
	retlw	D'25'		; for A/D of 127
	retlw	D'24'		; for A/D of 128
	retlw	D'24'		; for A/D of 129
	retlw	D'24'		; for A/D of 130
	retlw	D'23'		; for A/D of 131
	retlw	D'23'		; for A/D of 132
	retlw	D'23'		; for A/D of 133
	retlw	D'22'		; for A/D of 134
	retlw	D'22'		; for A/D of 135
	retlw	D'22'		; for A/D of 136
	retlw	D'21'		; for A/D of 137
	retlw	D'21'		; for A/D of 138
	retlw	D'21'		; for A/D of 139
	retlw	D'20'		; for A/D of 140
	retlw	D'20'		; for A/D of 141
	retlw	D'20'		; for A/D of 142
	retlw	D'19'		; for A/D of 143
	retlw	D'19'		; for A/D of 144
	retlw	D'19'		; for A/D of 145
	retlw	D'18'		; for A/D of 146
	retlw	D'18'		; for A/D of 147
	retlw	D'18'		; for A/D of 148
	retlw	D'17'		; for A/D of 149
	retlw	D'17'		; for A/D of 150
	retlw	D'17'		; for A/D of 151
	retlw	D'16'		; for A/D of 152
	retlw	D'16'		; for A/D of 153
	retlw	D'16'		; for A/D of 154
	retlw	D'15'		; for A/D of 155
	retlw	D'15'		; for A/D of 156
	retlw	D'15'		; for A/D of 157
	retlw	D'14'		; for A/D of 158
	retlw	D'14'		; for A/D of 159
	retlw	D'14'		; for A/D of 160
	retlw	D'13'		; for A/D of 161
	retlw	D'13'		; for A/D of 162
	retlw	D'12'		; for A/D of 163
	retlw	D'12'		; for A/D of 164
	retlw	D'12'		; for A/D of 165
	retlw	D'11'		; for A/D of 166
	retlw	D'11'		; for A/D of 167
	retlw	D'11'		; for A/D of 168
	retlw	D'10'		; for A/D of 169
	retlw	D'10'		; for A/D of 170
	retlw	D'10'		; for A/D of 171
	retlw	D'9'		; for A/D of 172
	retlw	D'9'		; for A/D of 173
	retlw	D'9'		; for A/D of 174
	retlw	D'8'		; for A/D of 175
	retlw	D'8'		; for A/D of 176
	retlw	D'7'		; for A/D of 177
	retlw	D'7'		; for A/D of 178
	retlw	D'7'		; for A/D of 179
	retlw	D'6'		; for A/D of 180
	retlw	D'6'		; for A/D of 181
	retlw	D'5'		; for A/D of 182
	retlw	D'5'		; for A/D of 183
	retlw	D'5'		; for A/D of 184
	retlw	D'4'		; for A/D of 185
	retlw	D'4'		; for A/D of 186
	retlw	D'4'		; for A/D of 187
	retlw	D'3'		; for A/D of 188
	retlw	D'3'		; for A/D of 189
	retlw	D'3'		; for A/D of 190
	retlw	D'2'		; for A/D of 191
	retlw	D'2'		; for A/D of 192
	retlw	D'1'		; for A/D of 193
	retlw	D'1'		; for A/D of 194
	retlw	D'0'		; for A/D of 195
	retlw	D'0'		; for A/D of 196
	retlw	D'0'		; for A/D of 197
	retlw	D'0'		; for A/D of 198

INTERRUPT; used for offset timeout. 5ms each interrupt and OFFS_COUNT1/2 are the counters to extend the time period 5ms/count 
	bcf		INTCON,TMR0IF	; flag cleared for timer0
; check offset flags for channel 1. If both zero, bypass decrementing counter for channel 1 
	movf	OFFS_1P,w	; positive offset flag
	btfss	STATUS,Z
	goto	DEC_T1
	movf	OFFS_1N,w	; negative offset flag
	btfsc	STATUS,Z
	goto	BY_DEC_T1
DEC_T1
	movf	OFFS_COUNT1,w	; counter for channel 1 check if zero
	btfss	STATUS,Z		; no decrement if 0 or decrease to 0
	decf	OFFS_COUNT1,f
BY_DEC_T1

; check offset flags for channel 2. If both zero, bypass decrementing counter for channel 2 
	movf	OFFS_2P,w	; positive offset flag
	btfss	STATUS,Z
	goto	DEC_T2
	movf	OFFS_2N,w	; negative offset flag
	btfsc	STATUS,Z
	goto	BY_DEC_T2
DEC_T2
	movf	OFFS_COUNT2,w	; counter for channel 1 check if zero
	btfss	STATUS,Z		; no decrement if 0 or decrease to 0
	decf	OFFS_COUNT2,f
BY_DEC_T2

	retfie	; end of interrupt

;........................................................

MAIN

; CWG
	movlb	D'13'			; bank 13, CWG
	bcf		CWG1CON0,G1EN	; CWG disabled 

; set inputs/outputs
	movlb	D'2'			; latch, bank 2
	clrf	LATC
	clrf	LATA
	clrf	LATB

; USB
	movlb	D'29'
	bcf		UCON,3			; USB off
	bcf		UCON,1
	
; weak pullups off
	movlb	D'4'			; bank4	WPUA/B
	clrf	WPUA
	clrf	WPUB

; set I/O
	movlb	D'1'			; bank1	TRISA/B/C

	movlw	B'00001011'	; 
	movwf	TRISA
	movlw	B'00110000'	; 
	movwf	TRISB
	movlw	B'11001111'	; 
	movwf	TRISC

; options
	movlw	B'00010111'		; weak pullups set via WPUA/B. TMR0 /256 Fosc/4 for 5ms/overflow
	movwf	OPTION_REG	; 
	bsf		INTCON,TMR0IE	; enable timer0

; analog inputs
	movlb	D'3'		; bank3	ANSELA/B/C
	movlw	B'00000000'	; 
	movwf	ANSELA
	movlw	B'00110000' ; AN10, AN11
	movwf	ANSELB
	movlw	B'11001111'	; AN4,5,6,7,8,9
	movwf	ANSELC

; oscillator
	movlb	D'1'	
	movlw	B'11111100'	; for 48MHz; 16MHz x 3PLL
	movwf	OSCCON	; osc

; timer2 set
	movlb	D'0'		; bank0	
	movlw	D'119'		; PWM  
	movwf	PR2			; PWM period register
; use prescaler of /4 and 48MHz clock 
; frequency = 1/ (333.333ns x (PR2 +1))
; PR2=119 for 25kHz

	movlw	B'00000001'	 	; /4 prescaler
	movwf	T2CON
	bsf		T2CON,2			; timer 2 on

; PWM set
	movlb	D'12'			; bank12  PWM	
	movlw	B'11100000'		; set PWM mode
	movwf	PWM1CON			; enable PWM1 operation
	clrf	PWM1DCL
	clrf	PWM1DCH			; 0 duty cycle, maximum duty at D119

; ensure the USB is off 
	movlb	D'29'
	bcf		UCON,3			; usb off

; interrupt on change settings
	movlb	D'07'			; bank 7
	movlw	B'00001011'		; IOC positive and negative edge for RA0,1,3
	movwf	IOCAP
	movwf	IOCAN
	movlb	D'0'			; bank 0

; timer1 set up
	movlw	B'11000001'		; uses internal 31kHz oscillator and /1 prescaler (bits 5,4) 
	movwf	T1CON			; 2.088s overflow and 32us per count

; Initial settings
	clrf	OFFS_COUNT1
	clrf	OFFS_COUNT2

; checking fans
; start fans using min pwm as stored. check for signal
; read fans running. If no fans, run setup. If fans running differs from fans stored, run setup 
; otherwise switch off pwm and continue with initial startup delay
; setup
; apply PWM, check for running fans. light fans on
; find starting PWM for fans (worst case PWM to start)  
; store fans connected and start pwm

; Read stored value for minimum pwm 
	movlw	H'07'			; ms address byte (H0701)
	movwf	READADDMS 
	movlw	H'01'
	movwf	READADDLS		; ls address byte
	call	READ			; 'w' has data for MIN_PWM
	movwf	MIN_PWM

; Read stored value for fans on
	movlw	H'07'			; ms address byte (H0701)
	movwf	READADDMS 
	movlw	H'00'
	movwf	READADDLS		; ls address byte
	call	READ			; 'w' has data for FANS
	movwf	FANS

NO_FAN_LOOP
; if minimum PWM is zero run setup
	movf	MIN_PWM,w
	btfsc	STATUS,Z
	goto	SETUP

; if FANS is zero run setup
	movf	FANS,w
	btfsc	STATUS,Z
	goto	SETUP

	movlw	D'5'			; was D'31'			; for 10s max time to start
	movwf	FAN_RUN			; counter

; start fans
; find connected fans
	movf	MIN_PWM,w
	movlb	D'12'			; bank12  PWM	
	clrf	PWM1DCL
	movwf	PWM1DCH			; apply stored duty cycle, maximum duty at D119
	movlb	D'0'			; bank 0

; fans should run
; wait 300ms, 8.25ms per count
; preload with FFFF-DBFF
	movlw	H'DB'			; 36 x 8.25ms=297ms
	call	DELAY

; clear IOC flags
	movlb	D'07'			; bank 7
	clrf	IOCAF
	movlb	D'0'			; bank 0	

; wait 300ms, 8.25ms per count
; preload with FFFF-DBFF
	movlw	H'DB'			; 36 x 8.25ms=297ms
	call	DELAY
	goto	SF1				; bypass DELAYX (Flashes fan LEDs)
; check fans at IOCAF,0,1,3 for IOC flags
START_FAN1
; preload with FFFF-DBFF
	movlw	H'86'			; 121 x 8.25ms=1s doubled in DELAYX (was H'ED' 18 x 8.25ms=148ms)
	call	DELAYX
SF1	
; check IOC flags
	movlb	D'07'			; bank 7
	movf	IOCAF,w			; flags
	movlb	D'0'			; bank 0
	movwf	TEMPS			; temporary

; check if fans run
	movf	TEMPS,w			; fans running flags
	btfss	STATUS,Z		; if zero, fans not running
	goto	FANS_ON11		; fan/s detected
	decfsz	FAN_RUN,f		; run until counts down to zero
	goto	START_FAN1
	goto	FANS_OFF1		; end of time to check

FANS_ON11
; read fans on
	movf	FANS,w			; store fans on
; if TEMPS=FANS continue. If not, redo check until time out
	xorwf	TEMPS,w
	btfsc	STATUS,Z
	goto	FANS_OFF1
	decfsz	FAN_RUN,f		; run until counts down to zero
	goto	START_FAN1

FANS_OFF1
; fans off
	movlb	D'12'			; bank12  PWM	
	clrf	PWM1DCL
	clrf	PWM1DCH			; off
	movlb	D'0'			; bank 0

; if TEMPS is zero then no fans
; bit 0 for fan2
; bit 1 for fan3
; bit 3 for fan1

	movf	TEMPS,w
	btfsc	STATUS,Z
	goto	SETUP			; no fans detected so run setup

; read fans on
				
	movf	FANS,w			; stored fans on

; if TEMPS=FANS continue. If not, run SETUP
	xorwf	TEMPS,w
	btfsc	STATUS,Z
	goto	START

SETUP; set up fans
; find connected fans
; start fans
	movlb	D'12'			; bank12  PWM	
	clrf	PWM1DCL
	movlw	D'96'			; PWM duty	
	movwf	PWM1DCH			; apply 80% duty cycle, maximum duty D120 gives 100%
; using less than 100% allows shunting PWM to sense for when no fans are used
	movlb	D'0'			; bank 0

; fans should run
	movlw	D'5'			; for 10s max time to start
	movwf	FAN_RUN			; counter

; wait 2s for each FAN_RUN count for fans to run
; preload with FFFF-0DFF
WAIT_T
	movlw	H'86'			; 121 x 8.25ms=1s doubled in DELAYX
	call	DELAYX
	decfsz	FAN_RUN,f		; totals 10s
	goto	WAIT_T

; clear IOC flags
	movlb	D'07'			; bank 7
	clrf	IOCAF
	movlb	D'0'			; bank 0

; check fans at IOCAF,0,1,3
; wait 300ms, 8.25ms per count
; preload with FFFF-DBFF
	movlw	H'DB'			; 36 x 8.25ms=297ms
	call	DELAY

; check IOC flags
	movlb	D'07'			; bank 7
	movf	IOCAF,w			; flags
	movlb	D'0'			; bank 0
	movwf	TEMPS			; temporary

; fans off
	movlb	D'12'			; bank12  PWM	
	clrf	PWM1DCL
	clrf	PWM1DCH			; off
	movlb	D'0'			; bank 0

; if TEMPS is zero then no fans
; bit 0 for fan2
; bit 1 for fan3
; bit 3 for fan1

	movf	TEMPS,w
	btfsc	STATUS,Z
	goto  	FAN_ERROR		; flash fan LEDs

; light the running fan LEDs
	movf	TEMPS,w
	movwf	FANS
	movlb	D'2'			; latch, bank 2
	clrf	LATA			; LEDs off
	clrf	LATC

	btfsc	FANS,0	
	bsf		LATA,4			; LED2 on with fan2
	btfsc	FANS,1
	bsf		LATA,5			; LED3 on with fan3
	btfsc	FANS,3
	bsf		LATC,4			; LED1 on with fan1
	movlb	D'0'			; bank 0

; wait for fans to stop
; fans should run
	movlw	D'5'			; for 10s max time to start
	movwf	FAN_RUN			; counter

; wait 2s each FAN_RUN count for fans to stop
; preload with FFFF-0DFF
WAIT_T2
	movlw	H'0D'			; 242 x 8.25ms=2s
	call	DELAY
	decfsz	FAN_RUN,f		; totals 10s
	goto	WAIT_T2

; test for minimum duty to run all connected fans. Start at a low value.
	movlw	D'9'
	movwf	MIN_PWM			; duty initially a low value at 7.5%. All fans would/should run at 20%

FAN_LOOP
; clear IOC flags
	movlb	D'07'			; bank 7
	clrf	IOCAF
; drive fans
	movlb	D'12'			; bank12  PWM
	clrf	PWM1DCL
	movf	MIN_PWM,w
	movwf	PWM1DCH			; apply duty cycle, maximum duty at D119
	movlb	D'0'			; bank 0

; fans should run or as duty increased
	movlw	D'31'			; for 10s max time to start
	movwf	FAN_RUN			; counter

; check fans at RA0,1,3
START_FAN3
; wait 300ms, 8.25ms per count
; preload with FFFF-DBFF
	movlw	H'DB'			; 36 x 8.25ms=297ms
	call	DELAY

; check IOC flags
	movlb	D'07'			; bank 7
	movf	IOCAF,w			; flags
	movlb	D'0'			; bank 0
	movwf	TEMPS

; check if fans run
	movf	TEMPS,w
	btfss	STATUS,Z		; if zero, fans not running
	goto	FANS_ON3		; fan(s) detected
	decfsz	FAN_RUN,f		; run until counts down to zero
	goto	START_FAN3

FANS_ON3
	movf	TEMPS,w
	btfsc	STATUS,Z		; if zero, no fans detected, so increase duty
	goto	NEXT_DUTY

; check if TEMPS matches FANS. If so minimum duty found that runs all that are connected.
	movf	TEMPS,w
	xorwf	FANS,w
	btfsc	STATUS,Z		; when equal end of minimum duty cycle search for fans to run and store
	goto	LOAD

NEXT_DUTY
; if no fans detected or not all fans detected that are connected, increase duty and test again
	incf	MIN_PWM,f
; if over D120, then error
	movlw	D'119'
	subwf	MIN_PWM,w
	btfsc	STATUS,C
	goto	FAN_ERROR		; fans don't run even with full duty
	goto	FAN_LOOP
	
LOAD; 
; add extra duty for startup run margin 
	incf	MIN_PWM,f
	incf	MIN_PWM,f
; if over D120, then error
	movlw	D'119'
	subwf	MIN_PWM,w
	btfsc	STATUS,C
	goto	FAN_ERROR		; error as needs maximum duty to run	
VALUES_IN
; Write  values to flash memory
	call	WRITE			; erase and write to flash memory

; blink fans on LEDs
	movlb	D'2'			; latch, bank 2
	clrf	LATA			; LEDs off
	clrf	LATC
	movlb	D'0'
; wait 300ms, 8.25ms per count
; preload with FFFF-DBFF
	movlw	H'DB'			; 8.25ms=297ms
	call	DELAY

	movlb	D'2'	
	btfsc	FANS,0	
	bsf		LATA,4			; LED2 on with fan2
	btfsc	FANS,1
	bsf		LATA,5			; LED3 on with fan3
	btfsc	FANS,3
	bsf		LATC,4			; LED1 on with fan1
	movlb	D'0'			; bank 0
	goto	START

; fans not connected
FAN_ERROR
; flash fan LEDs
	movlb	D'2'			; latch, bank 2
	clrf	LATA			; LEDs off
	clrf	LATB
	clrf	LATC
	movlb	D'0'			; bank 0

; sound buzzer 
	
	clrf	COUNT1
	movlw	H'02'
	movwf	COUNT2		; chirp period counters
CHIRP1
	movlb	D'2'		; latch, bank 2
	bsf		LATB,6		; high to piezo
	movlb	D'0'		; bank 0

	call	FREQ		; delay for frequency

	movlb	D'2'		; latch, bank 2
	bcf		LATB,6		; low to piezo
	movlb	D'0'		; bank 0

	call	FREQ		; delay for frequency

	decfsz	COUNT1,f
	goto	CHIRP1
	decfsz	COUNT2,f
	goto	CHIRP1

; flash off period
; wait 300ms, 8.25ms per count
; preload with FFFF-DBFF
	movlw	H'DB'			; 36 x 8.25ms=297ms
	call	DELAY
	movlb	D'2'			; latch, bank 2
	bsf		LATA,4			; LED2 on for fan2
	bsf		LATA,5			; LED3 on for fan3
	bsf		LATC,4			; LED1 on for fan1
	movlb	D'0'			; bank 0
; flash on period	 
; wait 300ms, 8.25ms per count
; preload with FFFF-DBFF
	movlw	H'DB'			; 36 x 8.25ms=297ms
	call	DELAY

; start fans
	movlb	D'12'			; bank12  PWM	
	clrf	PWM1DCL
	movlw	D'96'			; 80% PWM duty	
	movwf	PWM1DCH			; apply duty cycle, maximum duty at D119
	movlb	D'0'			; bank 0

; check IOC flags
	movlb	D'07'			; bank 7
	movf	IOCAF,w			; flags
	movlb	D'0'			; bank 0
	movwf	TEMPS

; check if fans run
	movf	TEMPS,w
	btfss	STATUS,Z		; if zero, fans not running
	goto	SETUP			; fan(s)now detected. setup for connected fans
	goto	FAN_ERROR

; ..........................................

START
; fans off
	movlb	D'12'			; bank12  PWM	
	clrf	PWM1DCL
	clrf	PWM1DCH			; off
	movlb	D'0'			; bank 0

	clrf	T_ALM			; temperature alarm flag

; light LEDs
	movlb	D'2'			; latch, bank 2
	clrf	LATA			; LEDs off
	clrf	LATC

	btfsc	FANS,0	
	bsf		LATA,4			; LED2 on with fan2
	btfsc	FANS,1
	bsf		LATA,5			; LED3 on with fan3
	btfsc	FANS,3
	bsf		LATC,4			; LED1 on with fan1
	movlb	D'0'			; bank 0


; start delay 5s
; wait 2s, 8.25ms per count
; preload with FFFF-00FF
	movlw	H'00'			; 255 x 8.25ms=2.1s 
	call	DELAY
; wait 2s, 8.25ms per count
	movlw	H'00'			; 255 x 8.25ms=2.1s 
	call	DELAY
; wait 1s, 8.25ms per count
	movlw	H'7F'			; 127 x 8.25ms=1s 
	call	DELAY

; initial states
; A/D
; ADCON0. bit 6-2 CHS<4:0>: Analog Channel Select bits

;00011 = AN3
;00100 = AN4
;00101 = AN5
;00110 = AN6
;00111 = AN7
;01000 = AN8
;01001 = AN9
;01010 = AN10
;01011 = AN11

LOOP ; code loops back to here when running normally

; Thermistor 1
; read 
	movlb	D'1'		; bank1	ADCON0/1/2
	movlw	B'00011100'	; channel 7
	movwf	ADCON0
	movlw	B'01110000'	; frc.  ms bit clear left justified
	movwf	ADCON1
	bsf		ADCON0,ADON	; A/D on
	movlb	D'0'		; bank 0
	call	ACQUIRE_AD	; ms result A_D_HI, ls byte in A_D_LOW
	movf	A_D_HI,w
	call	TEMP_CALC	; lookup table. Returns deg C value for thermistor
	movwf	NTC_TH1		; thermistor 1 degrees C (0-100 range)

; Thermistor 2
; read 
	movlb	D'1'		; bank1	ADCON0/1/2
	movlw	B'00100000'	; channel 8
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	movlb	D'0'		; bank 0
	call	ACQUIRE_AD	; ms result A_D_HI, ls byte in A_D_LOW
	movf	A_D_HI,w
	call	TEMP_CALC	; lookup table. Returns deg C value for thermistor
	movwf	NTC_TH2		; thermistor 2 degrees C (0-100 range)

; Threshold trim VR1
; read 
	movlb	D'1'		; bank1	ADCON0/1/2
	movlw	B'00100100'	; channel 9
	movwf	ADCON0
	movlw	B'11110000'	; frc.  ms bit set, right justified
	movwf	ADCON1
	bsf		ADCON0,ADON	; A/D on
	movlb	D'0'		; bank 0
	call	ACQUIRE_AD	; ms result A_D_HI, ls byte in A_D_LOW
	call	CALC
	movf	A_D_LOW,w
	movwf	THRESH		; VR1 value. Initial Threshold temperature

; Range trim VR2
; read 
	movlb	D'1'		; bank1	ADCON0/1/2
	movlw	B'00101100'	; channel 11
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	movlb	D'0'		; bank 0
	call	ACQUIRE_AD	; ms result A_D_HI, ls byte in A_D_LOW
	call	CALC		; get deg C value
	movf	A_D_LOW,w
	movwf	RANGE		; VR2 value. temperature range

; OVER_T, trim VR3
; read 
	movlb	D'1'		; bank1	ADCON0/1/2
	movlw	B'00101000'	; channel 10
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	movlb	D'0'		; bank 0
	call	ACQUIRE_AD	; ms result A_D_HI, ls byte in A_D_LOW
	call	CALC		; get deg C value
	movf	A_D_LOW,w
	movwf	OVER_T		; VR3 value. over temperature 

; read amplifier supply AC
; should be about 4.7V with ripple
; if below 3V (A/D D153), switch off relay

; read 
	movlb	D'1'		; bank1	ADCON0/1/2
	movlw	B'00010000'	; channel 4
	movwf	ADCON0
	movlw	B'01110000'	; frc.  ms bit clear = left justified
	movwf	ADCON1
	bsf		ADCON0,ADON	; A/D on
	movlb	D'0'		; bank 0
	call	ACQUIRE_AD	; ms result A_D_HI, ls byte in A_D_LOW
	movf	A_D_HI,w
	movwf	AC_DET		; AC detect

; read amplifier 1 output
; if 2.5V then good. Relay off if outside of between 2.25V (A/D D115) and 2.75V (A/D D140) 

; read 
	movlb	D'1'		; bank1	ADCON0/1/2
	movlw	B'00010100'	; channel 5
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	movlb	D'0'		; bank 0
	call	ACQUIRE_AD	; ms result A_D_HI, ls byte in A_D_LOW
	movf	A_D_HI,w
	movwf	AMP1_OFF	; offset

; read amplifier 2 output
; if 2.5V then good. Relay off if outside of between 2.25V (A/D D115) and 2.75V (A/D D140) 

; read 
	movlb	D'1'		; bank1	ADCON0/1/2
	movlw	B'00011000'	; channel 6
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	movlb	D'0'		; bank 0
	call	ACQUIRE_AD	; ms result A_D_HI, ls byte in A_D_LOW
	movf	A_D_HI,w
	movwf	AMP2_OFF	; offset

;..............................................................
; all A/D inputs read and processed

; AC VOLTS
; check amplifier AC volts. If above 2.5V then AC detected
	clrf	OFF_AC		; initially clear, could allow relay on
	movf	AC_DET,w	; AC voltage
	sublw	D'127'		; if above or below 2.5V
	btfsc	STATUS,C	; 
	bsf		OFF_AC,0	; off flag. Check before switching on relay

; AMPLIFIER OFFSET1

; check amplifier 1 offset. Typically would be 2.5V or between 3.5 and 1.5V = good. Otherwise offset detected
	movf	AMP1_OFF,w	; amplifier offset value
	sublw	D'76'		; if above or below 1.5V
	btfss	STATUS,C	; 
	goto	OVER2_1		; above 1.5V

; counter ready start
	bcf		OFFS_1P,0	; clear positive offset when negative set
	btfsc	OFFS_1N,0	; if set bypass
	goto	CK_TIMCH1
	movlw	D'15'		; 75ms
	movwf	OFFS_COUNT1	; counter for channel 1 timer overflows
	bsf		INTCON,GIE	; enable interrupt
	bsf		OFFS_1N,0	; offset negative flag. Check before switching on relay

	goto	CK_OFF2		; amplifier 2

OVER2_1; check for above 3.5V
	movf	AMP1_OFF,w	; amplifier offset
	sublw	D'178'		; if above or below 3.5V
	btfsc	STATUS,C	; 
	goto	NO_OFF1

; counter ready start
; start counter and timer0
	bcf		OFFS_1N,0	; clear negative offset when positive set
	btfsc	OFFS_1P,0	; if set bypass
	goto	CK_TIMCH1
	movlw	D'15'		; 75ms
	movwf	OFFS_COUNT1	; counter for channel 1 timer overflows
	bsf		INTCON,GIE	; enable interrupt
	bsf		OFFS_1P,0	; offset positive flag
	goto	CK_OFF2		; amplifier 2

NO_OFF1 ; no offset so clear + and - offset flags
	clrf	OFFS_1N
	clrf	OFFS_1P
	clrf	OFFS_COUNT1

; if timed out and any offset flags are set, set offset flag
CK_TIMCH1	; timer for channel 1	
; Check timeout for channel 1
	movf	OFFS_COUNT1,w	; when zero check if both + or - offset flags are set. If so, set offset flag for amp1
	btfss	STATUS,Z
	goto	CK_OFF2			; check next amplifier 2/channel 2
; check both flags
	movf	OFFS_1N,w
	btfss	STATUS,Z
	goto	AMP1_SET
	movf	OFFS_1P,w
	btfss	STATUS,Z
	goto	AMP1_SET
; both zero so clear all flags
	clrf	OFF_OFFS1	; offset flag for amplifier 1
	goto	CLR_OFFS1	; clear other + and =flags
AMP1_SET; set amplifier flag and clear + and - flags
	bsf		OFF_OFFS1,0	; flag set
CLR_OFFS1
	clrf	OFFS_1P		; positive
	clrf	OFFS_1N		; negative


CK_OFF2
; AMPLIFIER OFFSET2
; check amplifier 2 offset. Typically would be 2.5V or between 3.5 and 1.5V = good. Otherwise offset detected
	movf	AMP2_OFF,w	; amplifier offset value
	sublw	D'76'		; if above or below 1.5V
	btfss	STATUS,C	; 
	goto	OVER2_2		; above 1.5V

; counter ready start
	bcf		OFFS_2P,0	; clear positive offset when negative set
	btfsc	OFFS_2N,0	; if set bypass
	goto	CK_TIMCH2
	movlw	D'15'		; 75ms
	movwf	OFFS_COUNT2	; counter for channel 1 timer overflows
	bsf		INTCON,GIE	; enable interrupt
	bsf		OFFS_2N,0	; offset negative flag. Check before switching on relay
	goto	TEMP_RUN	; temperature run

OVER2_2; check for above 3.5V
	movf	AMP2_OFF,w	; amplifier offset
	sublw	D'178'		; if above or below 3.5V
	btfsc	STATUS,C	; 
	goto	NO_OFF2

; counter ready start
; start counter and timer0
	bcf		OFFS_2N,0	; clear negative offset when positive set
	btfsc	OFFS_2P,0	; if set bypass
	goto	CK_TIMCH2
	movlw	D'15'		; 75ms
	movwf	OFFS_COUNT2	; counter for channel 2 timer overflows
	bsf		INTCON,GIE	; enable interrupt
	bsf		OFFS_2P,0	; offset positive flag
	goto	TEMP_RUN	; temperature run

NO_OFF2 ; no offset so clear + and - offset flags
	clrf	OFFS_2N
	clrf	OFFS_2P
	clrf	OFFS_COUNT2

; if timed out and any offset flags are set, set offset flag
CK_TIMCH2	; timer for channel 2	
; Check timeout for channel 2
	movf	OFFS_COUNT2,w	; when zero check if both + or - offset flags are set. If so, set offset flag for amp2
	btfss	STATUS,Z
	goto	TEMP_RUN		; temperature run
; check both flags
	movf	OFFS_2N,w
	btfss	STATUS,Z
	goto	AMP2_SET
	movf	OFFS_2P,w
	btfss	STATUS,Z
	goto	AMP2_SET
; both zero so clear all flags
	clrf	OFF_OFFS2	; offset flag for amplifier 2
	goto	CLR_OFFS2	; clear other + and =flags
AMP2_SET; set amplifier flag and clear + and - flags
	bsf		OFF_OFFS2,0	; flag set
CLR_OFFS2
	clrf	OFFS_2P		; positive
	clrf	OFFS_2N		; negative


TEMP_RUN; temperature run

; Flags and temperature
; switch on relay if not over temperature and offset and AC voltage flags are clear
; flags (OFF_OFF1, OFF_OFF2, OFF_AC) if all are zero then can allow relay on
; Check over temperature comparing NTC_TH1 and NTC_TH2 with OVER_T setting 
; OVER_T (VR3) over temperature comparison with NTC_TH1 and NTC_TH2.
; use highest temperature thermistor
	
	movf 	OFF_AC,w	; AC voltage flag 
	btfss	STATUS,Z
	goto	ALARM_NOFAN	; AC off

	movf 	OFF_OFFS1,w	; offset flag for amplifier 1
	btfss	STATUS,Z
	goto	ALARM_NOFAN	; offset alarm
	movf 	OFF_OFFS2,w	; offset flag for amplifier 2
	btfss	STATUS,Z
	goto	ALARM_NOFAN		; offset alarm
	
; if no ALARM, then all flags are zero

; find maximum temperature thermistor
	movf	NTC_TH1,w
	subwf	NTC_TH2,w
	btfss	STATUS,C	; if C=0 then NTC_TH1>NTC_TH2
	goto	USE_TH1
USE_TH2
	movf 	NTC_TH2,w	
	movwf	TEMP
	btfss	T_ALM,0		; if over temperature initiated add hysteresis
	goto	CF2
	movlw	D'4'		; hysteresis of 4 degrees C
	addwf	TEMP,w
CF2	subwf	OVER_T,w
	btfss	STATUS,C
	goto	ALARM1
	goto	RUN_FAN_CONTROL

USE_TH1
	movf 	NTC_TH1,w	
	movwf	TEMP
	btfss	T_ALM,0		; if over temperature initiated add hysteresis
	goto	CF1
	movlw	D'4'		; hysteresis of 4 degrees C
	addwf	TEMP,w
CF1	subwf	OVER_T,w
	btfsc	STATUS,C
	goto	RUN_FAN_CONTROL

ALARM1
; check if the same
	btfsc	STATUS,Z
	goto	RUN_FAN_CONTROL
	bsf		T_ALM,0
	goto 	ALARM		; over temperature

RUN_FAN_CONTROL

; no relay switch on if over temperature flag is set
	btfsc	T_ALM,0
	goto	BY_ON	
; switch on relay
	movlb	D'2'		; latch, bank 2
	bsf		LATB,7		; relay on
	movlb	D'0'		; bank 0

BY_ON
	clrf	T_ALM		; clear flag for when not in over temperature
; if not over temp, run fan control
; minimum duty at NTC_TH1 or NTC_TH2 = THRESH
; maximum duty at NTC_TH1 or NTC_TH2 = THRESH + RANGE
; calculation (maximum duty (120) - MIN_PWM = pwm duty variation min to max (DUTY_X)
; RANGE = temperature range for the duty variation
; increase duty by duty variation (DUTY_X)/range per degree
; so multiply duty variation (DUTY_X) by NTC_TH1 or NTC_TH2 above THRESH and divide by RANGE. Then add MIN_PWM for duty
; subtract THRESH from NTC_THx, if minus, then no fan speed required
; if plus then use subtracted value (NTC_THx - THRESH) multiplied by duty variation(DUTY_X)and divide by RANGE. Add MIN_PWM 

; find maximum temperature thermistor
	movf	NTC_TH1,w
	subwf	NTC_TH2,w
	btfss	STATUS,C	; if C=0 then NTC_TH1>NTC_TH2
	goto	USE_TH1X
USE_TH2X
	movf	THRESH,w
	subwf	NTC_TH2,w
	btfss	STATUS,C
	goto	NO_CONTROL	; no control needed so switch off duty
	goto	STORE_SUBT
USE_TH1X
	movf	THRESH,w
	subwf	NTC_TH1,w
	btfss	STATUS,C
	goto	NO_CONTROL	; no control needed so switch off duty
STORE_SUBT	; store subtracted value
	movwf	TEMPS		; temperature above THRESHOLD

; if TEMPS > or = RANGE then use 120
	movf	TEMPS,w
	subwf	RANGE,w
	btfss	STATUS,C
	goto	USE_120
	btfsc	STATUS,Z
	goto	USE_120

; 120 - MIN_PWM = DUTY_X
	movf	MIN_PWM,w
	sublw	D'119'
	movwf	DUTY_X
	btfss	STATUS,C
	goto	USE_120		; if negative use full pwm duty
	btfsc	STATUS,Z
	goto	USE_120		; if zero use 120

; if range is 0, bypass calculation and use 120 for duty (otherwise calculation would divide by 0)
	movf	RANGE,w
	btfsc	STATUS,Z
	goto	USE_120

; Calculate: (NTC_THx - THRESH)= TEMPS multiplied by duty variation(DUTY_X)and divide by RANGE 

; load values; TEMPS
	movf	TEMPS,w		; ls byte
	movwf	AARGB2
	clrf	AARGB1
	clrf	AARGB0		; ms byte

; Multiply by DUTY_X
	movf	DUTY_X,w
	movwf	BARGB2
	clrf	BARGB1
	clrf	BARGB0
	call	FXM2424U		; multiply
; shift result for division	
	movf	AARGB2,w		; ms of multiplication
	movwf	AARGB0
	movf	AARGB3,w		; ls byte
	movwf	AARGB1
	movf	AARGB4,w		; ms of multiplication
	movwf	AARGB2
	movf	AARGB5,w		; ls byte
	movwf	AARGB3
; Divide by RANGE
	clrf	BARGB0
	clrf	BARGB1
	clrf  	BARGB2
	movf	RANGE,w
	movwf	BARGB3	
	call	FXD3232U		; divide

; result in AARGB3

; if AARGB0, AARGB1, AARGB2 are not 0 (ie result >255) use D120
	movf	AARGB0,w
	btfss	STATUS,Z
	goto	USE_120	
	movf	AARGB1,w
	btfss	STATUS,Z
	goto	USE_120	
	movf	AARGB2,w
	btfss	STATUS,Z
	goto	USE_120	

; add MIN_PWM
; if >119 then set at 120 fixed value
	movf	AARGB3,w
	addwf	MIN_PWM,w
	movwf	AARGB3		; store
	sublw	D'119'
	btfss	STATUS,C
	goto	USE_120
	movf	AARGB3,w
	goto	DRV_FANS
USE_120
	movlw	D'120'		; use 120 if over
	
; drive fans
DRV_FANS
	movlb	D'12'			; bank12  PWM
	clrf	PWM1DCL
	movwf	PWM1DCH			; apply duty cycle from calculated at AARGB3, maximum duty at D119
	movlb	D'0'			; bank 0
	goto	LOOP

NO_CONTROL
; switch off duty
	movlb	D'12'			; bank12  PWM
	clrf	PWM1DCL
	clrf	PWM1DCH			; clear duty cycle
	movlb	D'0'
	goto 	LOOP

ALARM

; run fans at max
; start fans
	movlb	D'12'		; bank12  PWM	
	clrf	PWM1DCL
	movlw	D'120'		; maximum PWM duty	
	movwf	PWM1DCH		; apply duty cycle, maximum duty at D119
	movlb	D'0'		; bank 0
ALARM_NOFAN

; switch off relay
	movlb	D'2'		; latch, bank 2
	bcf		LATB,7		; relay off
	movlb	D'0'		; bank 0

; bypass buzzer if AC loss detected
	movf 	OFF_AC,w	; AC voltage flag 
	btfss	STATUS,Z
	goto	LOOP

; sound buzzer momentarily on each return to here
	bcf		INTCON,GIE	; interrupt off
	clrf	COUNT1
	movlw	H'02'
	movwf	COUNT2		; chirp period counters
CHIRP
	movlb	D'2'		; latch, bank 2
	bsf		LATB,6		; high to piezo
	movlb	D'0'		; bank 0

	call	FREQ		; delay for frequency

	movlb	D'2'		; latch, bank 2
	bcf		LATB,6		; low to piezo
	movlb	D'0'		; bank 0

	call	FREQ		; delay for frequency

	decfsz	COUNT1,f
	goto	CHIRP
	decfsz	COUNT2,f
	goto	CHIRP
	
; Off period, preload with FFFF-F3FF
	movlw	H'F3'			; 32 x 8.25ms=99ms
	call	DELAY

	goto	LOOP		; chirp off; continue and check all inputs	

;.................................................................................
; subroutines

DELAY; uses timer1 running at 31kHz no prescaler and preloading for delay FFFF-xxFF. xx=8.25ms per count (2s max)		
; stop timer
	bcf		T1CON,0
; preload timer 1
	movwf	TMR1H			; value loaded in w before call (ms byte)
	movlw	H'FF'
	movwf	TMR1L
DELAY1
	bsf		T1CON,0			; start timer1
	bcf		PIR1,TMR1IF
TMR_LOOP1
	btfss	PIR1,TMR1IF
	goto	TMR_LOOP1
	return

FREQ ; piezo drive frequency target at 4kHz with 250us period, so timer at 125us for each half cycle.
; closest value is to use counter set at 4 for 129us and 3.875kHz
; FFFF-FFxx. xx=32.25us per count
	movlw	H'FC'	

DELAY2; uses timer1 running at 31kHz no prescaler and preloading for delay FFFF-FFxx. xx=32.25us per count (8.25ms max)		
; stop timer

	bcf		T1CON,0
; preload timer 1
	movwf	TMR1L	; value loaded in w before call (ls byte)
	movlw	H'FF'
	movwf	TMR1H
	goto	DELAY1	

DELAYX; flash on period dependent on the W register value at call DELAYX
	movwf	TEMP

; flash fan LEDs
	movlb	D'2'			; latch, bank 2
	bsf		LATA,4			; LED2 on for fan2
	bsf		LATA,5			; LED3 on for fan3
	bsf		LATC,4			; LED1 on for fan1
	movlb	D'0'			; bank 0

	movf	TEMP,w
	call	DELAY

	movlb	D'2'			; latch, bank 2
	bcf		LATA,4			; LED2 off for fan2
	bcf		LATA,5			; LED3 off for fan3
	bcf		LATC,4			; LED1 off for fan1
	movlb	D'0'			; bank 0
	movf	TEMP,w
	call	DELAY
	return

; subroutine to wait for A/D conversion 
ACQUIRE_AD

; wait >8us to charge input capacitance 

	movlw	H'F9'	; 64us
	movwf	COUNT1
DELC
	decfsz	COUNT1,f
	goto	DELC
AD_1
	movlb	D'1'		; bank 1
	bsf		ADCON0,1	; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,1	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	movf	ADRESL,w
	movwf	A_D_LOW		; least significant bits
	movf	ADRESH,w
	movlb	D'0'		; bank0
	movwf	A_D_HI		; A/D conversion high byte	
	return

; subroutine to calculate degrees C value from trim pots
CALC
; subtract D558 (10-bit A/D value for 2.73V) multiply by 500 (5V range)then divide by 1023 (A/D range) to convert from A/D value
; to degrees C. This is to compare with NTC_TH values directly.
; Note trim pot voltage is set so 2.73V=273K. So 10 degrees C is 283K and so is set at 2.83V 

; subtract 558D (H22E)
	movlw	H'2'			; high byte
	subwf	A_D_HI,f		; take from revised
	movlw	H'2E'			; low byte
	subwf	A_D_LOW,f		; revised
	btfss	STATUS,C
	decf	A_D_HI,f		; decrease if required
	btfss	A_D_HI,7		; if set then H22E> A_D value so set at 0
	goto	MULT_1
	clrf	A_D_LOW
	clrf	A_D_HI
	return

MULT_1
; so multiply 500
; load values
	movf	A_D_LOW,w		; ls byte
	movwf	AARGB2
	movf	A_D_HI,w		; ms byte
	movwf	AARGB1
	clrf	AARGB0			; ms byte

; Multiply by decimal 500 = 1F4	
	movlw	H'F4'
	movwf	BARGB2
	movlw	H'1'
	movwf	BARGB1
	clrf	BARGB0
	call	FXM2424U		; multiply
; shift result for division	
	movf	AARGB2,w		; ms of multiplication
	movwf	AARGB0
	movf	AARGB3,w		; 
	movwf	AARGB1
	movf	AARGB4,w		; 
	movwf	AARGB2
	movf	AARGB5,w		; ls byte
	movwf	AARGB3
; Divide by D1023 3FFH
	clrf	BARGB0
	clrf	BARGB1
	movlw	H'3'
	movwf  	BARGB2
	movlw	H'FF'
	movwf	BARGB3	
	call	FXD3232U		; divide

; result in AARGB2,	AARGB3,w
	movf	AARGB3,w
	movwf	A_D_LOW			; revised low byte
	movf	AARGB2,w
	movwf	A_D_HI			; revised high byte
	return

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
; read data memory
READ; 'w' has read data
	bcf		 INTCON,GIE 	; Disable interrupts so required sequences will execute properly
	movlb	D'3'				;  bank 3 
	movf	READADDMS,w 
	movwf 	PMADRH		; ms Byte of Program Address to read
	movf	READADDLS,w
	movwf 	PMADRL 		; ls Byte of Program Address to read
	bcf		PMCON1,CFGS	; avoid configuration space
	bsf	 	PMCON1,RD 	; Read
	nop 					
	nop
; memory is read in second cycle PM read instruction
;	movf	PMDATH,w 		; ms Byte of Program data 
;	movwf	TEMP_VAL
	movf	PMDATL,w 		; ls Byte of Program data
	movlb	D'0'			; bank 0
	return
;.............................................

; write to data memory
WRITE

; erase first before write
	bcf		 INTCON,GIE 	; Disable interrupts so required sequences will execute properly
	movlb	 D'03'
	movlw	 D'07'			 ; Load initial program memory address
	movwf	 PMADRH ;
	movlw	 D'00'
	movwf	 PMADRL ;
;
	bcf		 PMCON1,CFGS ; Not configuration space
	bsf		 PMCON1,FREE ; Enable erase
	bsf		 PMCON1,WREN ; Enable writes
	
	movlw	 H'55'		 	; Start of required write sequence:
	movwf	 PMCON2 		; Write 55h
	movlw	 H'AA' ;
	movwf	 PMCON2		 ; Write AAh
	bsf		 PMCON1,WR	 ; Set WR bit to begin write
	nop
	nop
	bcf		 PMCON1,WREN ; Disable writes

	movlw	 D'07'		 ; Load initial program memory address
	movwf	 PMADRH ;
	movlw	 D'00'
	movwf	 PMADRL ;

	bcf		 PMCON1,CFGS ; Not configuration space
	bsf		 PMCON1,WREN ; Enable writes
	bsf		 PMCON1,LWLO ; Only Load Write Latches

;	movf	 FANS0,w
	movlw	 H'3F'
	movwf	 PMDATH ;
	movf	 FANS,w
	movwf	 PMDATL ;

	movlw	 H'55'		 ; Start of required write sequence:
	movwf	 PMCON2 	 ; Write 55h
	movlw	 H'AA' ;
	movwf	 PMCON2		 ; Write AAh
	bsf		 PMCON1,WR 	 ; Set WR bit to begin write
	nop					 ; nop instructions are forced as processor loads program memory write latches
	nop ;
	incf	 PMADRL,f	 ; Still loading latches Increment address
	
;	movf	 MIN_PWM0,w
	movlw	 H'3F'	
	movwf	 PMDATH ;
	movf	 MIN_PWM,w
	movwf	 PMDATL ;

	movlw	 H'55'		 ; Start of required write sequence:
	movwf	 PMCON2 	 ; Write 55h
	movlw	 H'AA' ;
	movwf	 PMCON2	     ; Write AAh
	bsf		 PMCON1,WR   ; Set WR bit to begin write
	nop					 ; nop instructions are forced as processor loads program memory write latches
	nop ;
	

START_WRITE
	bcf		PMCON1,LWLO ; No more loading latches - Start Flash program
; memory write
	movlw	H'55'		; Start of required write sequence:
	movwf	PMCON2 		; Write 55h
	movlw	H'AA' ;
	movwf	PMCON2 		; Write AAh
	bsf		PMCON1,WR 	; Set WR bit to begin write
	nop					; NOP instructions are forced as processor writes all the program memory write latches simultaneously to program memory.
	nop	  
	bcf		PMCON1,WREN ; Disable writes
	movlb	 D'0'
	return

; *********************************
; 24x24 Bit Unsigned Fixed Point Multiply 24x24 -> 48
; Input: 24 bit unsigned fixed point multiplicand in AARGB0,1,2
; 24 bit unsigned fixed point multiplier in BARGB0,1,2
; Use: CALL FXM2424U
; Output: 48 bit unsigned fixed point product in AARGB0
; Result: AARG <-- AARG x BARG
; Max Timing: 9+501+2 = 512 clks
; Min Timing: 9+150 = 159 clks

FXM2424U
	CLRF 	AARGB3 ; clear partial product
	CLRF 	AARGB4
	CLRF 	AARGB5
	MOVF 	AARGB0,W
	MOVWF 	TEMPB0
	MOVF 	AARGB1,W
	MOVWF 	TEMPB1
	MOVF 	AARGB2,W
	MOVWF 	TEMPB2

	MOVLW 	H'08'
	MOVWF 	LOOPCOUNT
LOOPUM2424A
	RRF 	BARGB2,F
	BTFSC 	STATUS,C
	GOTO 	ALUM2424NAP
	DECFSZ 	LOOPCOUNT,F
	GOTO	LOOPUM2424A
	MOVWF 	LOOPCOUNT
LOOPUM2424B
	RRF 	BARGB1,F
	BTFSC 	STATUS,C
	GOTO	BLUM2424NAP
	DECFSZ	LOOPCOUNT,F
	GOTO	LOOPUM2424B
	MOVWF	LOOPCOUNT
LOOPUM2424C
	RRF		BARGB0,F
	BTFSC 	STATUS,C
	GOTO 	CLUM2424NAP
	DECFSZ 	LOOPCOUNT,F
	GOTO 	LOOPUM2424C
	CLRF 	AARGB0
	CLRF 	AARGB1
	CLRF 	AARGB2
	RETLW 	0x00
CLUM2424NAP
	BCF 	STATUS,C
	GOTO 	CLUM2424NA
BLUM2424NAP
	BCF 	STATUS,C
	GOTO 	BLUM2424NA
ALUM2424NAP
	BCF 	STATUS,C
	GOTO 	ALUM2424NA
ALOOPUM2424
	RRF 	BARGB2,F
	BTFSS 	STATUS,C
	GOTO 	ALUM2424NA
	MOVF 	TEMPB2,W
	ADDWF 	AARGB2,F
	MOVF 	TEMPB1,W
	BTFSC 	STATUS,C
	INCFSZ 	TEMPB1,W
	ADDWF 	AARGB1,F
	MOVF 	TEMPB0,W
	BTFSC 	STATUS,C
	INCFSZ 	TEMPB0,W
	ADDWF	AARGB0,F
ALUM2424NA
	RRF 	AARGB0,F
	RRF 	AARGB1,F
	RRF 	AARGB2,F
	RRF 	AARGB3,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	ALOOPUM2424
	MOVLW 	H'08'
	MOVWF 	LOOPCOUNT
BLOOPUM2424
	RRF 	BARGB1,F
	BTFSS 	STATUS,C
	GOTO 	BLUM2424NA
	MOVF 	TEMPB2,W
	ADDWF 	AARGB2,F
	MOVF 	TEMPB1,W
	BTFSC 	STATUS,C
	INCFSZ 	TEMPB1,W
	ADDWF 	AARGB1,F
	MOVF 	TEMPB0,W
	BTFSC 	STATUS,C
	INCFSZ 	TEMPB0,W
	ADDWF 	AARGB0,F
BLUM2424NA
	RRF 	AARGB0,F
	RRF 	AARGB1,F
	RRF 	AARGB2,F
	RRF 	AARGB3,F
	RRF 	AARGB4,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	BLOOPUM2424
	MOVLW 	H'08'
	MOVWF 	LOOPCOUNT
CLOOPUM2424
	RRF 	BARGB0,F
	BTFSS 	STATUS,C
	GOTO 	CLUM2424NA
	MOVF 	TEMPB2,W
	ADDWF 	AARGB2,F
	MOVF 	TEMPB1,W
	BTFSC 	STATUS,C
	INCFSZ 	TEMPB1,W
	ADDWF 	AARGB1,F
	MOVF 	TEMPB0,W
	BTFSC 	STATUS,C
	INCFSZ 	TEMPB0,W
	ADDWF 	AARGB0,F
CLUM2424NA
	RRF 	AARGB0,F
	RRF 	AARGB1,F
	RRF 	AARGB2,F
	RRF 	AARGB3,F
	RRF 	AARGB4,F
	RRF 	AARGB5,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	CLOOPUM2424
	return

; ********************************
; 32/32 Bit Unsigned Fixed Point Divide 32/32 -> 32.32
; Input: 32 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2,AARGB3
; 32 bit unsigned fixed point divisor in BARGB0, BARGB1, BARGB2, BARGB3
; Use: CALL FXD3232U
; Output: 32 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2,AARGB3
; 32 bit unsigned fixed point remainder in REMB0, REMB1, REMB2, REMB3
; Result: AARG, REM <-- AARG / BARG
; Max Timing: 4+1025+2 = 1031 clks
; Max Timing: 4+981+2 = 987 clks
; PM: 4+359+1 = 364 DM: 13
FXD3232U
	CLRF 	REMB0
	CLRF	REMB1
	CLRF 	REMB2
	CLRF 	REMB3
	call	UDIV3232L
	return

UDIV3232L 
; Max Timing: 24+6*32+31+31+6*32+31+31+6*32+31+31+6*32+31+16 = 1025 clks
; Min Timing: 24+6*31+30+30+6*31+30+30+6*31+30+30+6*31+30+3 = 981 clks
; PM: 359 DM: 13
	CLRF 	TEMP
	RLF 	AARGB0,W
	RLF 	REMB3,F
	MOVF 	BARGB3,W
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	RLF 	AARGB0,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232A 
	RLF 	AARGB0,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB0,0
	GOTO 	UADD22LA
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS	STATUS,C
	INCFSZ	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22LA

UADD22LA 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22LA 
	RLF		AARGB0,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	LOOPU3232A
	RLF 	AARGB1,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB0,0
	GOTO 	UADD22L8
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22L8

UADD22L8 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22L8 
	RLF 	AARGB1,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232B 
	RLF 	AARGB1,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB1,0
	GOTO 	UADD22LB
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22LB

UADD22LB 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22LB 
	RLF 	AARGB1,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	LOOPU3232B
	RLF 	AARGB2,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS	AARGB1,0
	GOTO 	UADD22L16
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22L16

UADD22L16 	
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF	TEMP,F

UOK22L16
	RLF		AARGB2,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232C 
	RLF 	AARGB2,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB2,0
	GOTO 	UADD22LC
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22LC

UADD22LC
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22LC 
	RLF 	AARGB2,F
	DECFSZ 	LOOPCOUNT,F
	GOTO 	LOOPU3232C
	RLF 	AARGB3,W
	RLF 	REMB3,F
	RLF 	REMB2,F
	RLF 	REMB1,F
	RLF 	REMB0,F
	RLF 	TEMP,F
	MOVF 	BARGB3,W
	BTFSS 	AARGB2,0
	GOTO 	UADD22L24
	SUBWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB2,W
	SUBWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB1,W
	SUBWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSS 	STATUS,C
	INCFSZ 	BARGB0,W
	SUBWF 	REMB0,F
	CLRW
	BTFSS 	STATUS,C
	MOVLW 	H'1'
	SUBWF 	TEMP,F
	GOTO 	UOK22L24

UADD22L24 
	ADDWF 	REMB3,F
	MOVF 	BARGB2,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB2,W
	ADDWF 	REMB2,F
	MOVF 	BARGB1,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB1,W
	ADDWF 	REMB1,F
	MOVF 	BARGB0,W
	BTFSC 	STATUS,C
	INCFSZ 	BARGB0,W
	ADDWF 	REMB0,F
	CLRW
	BTFSC 	STATUS,C
	MOVLW 	H'1'
	ADDWF 	TEMP,F

UOK22L24 
	RLF 	AARGB3,F
	MOVLW 	H'7'
	MOVWF 	LOOPCOUNT

LOOPU3232D 
    RLF  	AARGB3,W
    RLF    	REMB3,F
    RLF    	REMB2,F
    RLF    	REMB1,F
    RLF    	REMB0,F
    RLF    	TEMP,F
    MOVF   	BARGB3,W
 	BTFSS  	AARGB3,0
   	GOTO   	UADD22LD

    SUBWF  	REMB3,F
  	MOVF   	BARGB2,W
    BTFSS   STATUS,C
    INCFSZ  BARGB2,W
    SUBWF   REMB2,F
    MOVF    BARGB1,W
    BTFSS   STATUS,C
    INCFSZ  BARGB1,W
    SUBWF   REMB1,F
    MOVF    BARGB0,W
    BTFSS   STATUS,C
    INCFSZ  BARGB0,W
    SUBWF   REMB0,F
    CLRW
    BTFSS   STATUS,C
    MOVLW   H'1'
    SUBWF   TEMP,F
    GOTO    UOK22LD

UADD22LD      
	ADDWF   REMB3,F
    MOVF    BARGB2,W
    BTFSC   STATUS,C
    INCFSZ  BARGB2,W
    ADDWF   REMB2,F
    MOVF    BARGB1,W
    BTFSC   STATUS,C
    INCFSZ  BARGB1,W
    ADDWF   REMB1,F
    MOVF    BARGB0,W
    BTFSC   STATUS,C
    INCFSZ  BARGB0,W
    ADDWF   REMB0,F
    CLRW
    BTFSC   STATUS,C
    MOVLW   H'1'
    ADDWF   TEMP,F
        
UOK22LD
	RLF     AARGB3,F

   	DECFSZ  LOOPCOUNT, F
    GOTO    LOOPU3232D

    BTFSC   AARGB3,0
    GOTO    UOK22L
    MOVF    BARGB3,W
	ADDWF   REMB3,F
    MOVF    BARGB2,W
    BTFSC   STATUS,C
    INCFSZ  BARGB2,W
    ADDWF   REMB2,F
    MOVF    BARGB1,W
    BTFSC   STATUS,C
    INCFSZ  BARGB1,W
    ADDWF   REMB1,F
    MOVF    BARGB0,W
    BTFSC   STATUS,C
    INCFSZ  BARGB0,W
    ADDWF   REMB0,F

UOK22L

	RETURN


	end

